<?php
/**
 * Plugin Name: Speak Easy
 * Plugin URI:  https://persoft.de/tipps-tricks/speak-easy/
 * Description: Reads website text aloud. Also supports foreign language plugins like GTranslate.
 * Version:     1.4.0
 * Author:      PERSoft
 * Author URI:  https://persoft.de
 * Text Domain: speak-easy
 * Domain Path: /languages
 */

/* -------------------------------------------------------------
 * Sprachdateien laden
 * ------------------------------------------------------------- */
add_action('plugins_loaded', function () {
    load_plugin_textdomain('speak-easy', false, dirname(plugin_basename(__FILE__)) . '/languages/');
});

/* -------------------------------------------------------------
 * Aktivierung: Standardwerte setzen
 * ------------------------------------------------------------- */
register_activation_hook(__FILE__, 'speak_easy_activate');
function speak_easy_activate() {
    if (get_option('speak_easy_custom_css') === false) {
        $css_file = plugin_dir_path(__FILE__) . 'css/speakeasy.css';
        $default_css = file_exists($css_file) ? file_get_contents($css_file) : '';
        update_option('speak_easy_custom_css', $default_css);
    }
    if (get_option('speak_easy_button_title') === false) update_option('speak_easy_button_title', null);
    if (get_option('speak_easy_lang_source') === false) update_option('speak_easy_lang_source', 'de-DE');
    if (get_option('speak_easy_bilder_erfassen') === false) update_option('speak_easy_bilder_erfassen', 0);
    if (get_option('speak_easy_word_replacements') === false) update_option('speak_easy_word_replacements', array());
}

/* -------------------------------------------------------------
 * CSS laden (DB oder Datei)
 * ------------------------------------------------------------- */
function speak_easy_get_custom_css() {
    $css = get_option('speak_easy_custom_css', '');
    if (empty(trim($css))) {
        $css_file = plugin_dir_path(__FILE__) . 'css/speakeasy.css';
        if (file_exists($css_file)) {
            $css = file_get_contents($css_file);
        }
    }
    return $css;
}

/* -------------------------------------------------------------
 * Frontend Shortcode
 * ------------------------------------------------------------- */
function speaker_button_shortcode() {
    $title = get_option('speak_easy_button_title') ?: __('Read text / Stop reading', 'speak-easy');
    $title = esc_attr($title);
    ob_start(); ?>
    <div id="speaker" class="hidden" aria-hidden="false">
        <button title="<?php echo $title; ?>" class="responsivevoice-button" id="speaker-button1" type="button" value="Play" aria-pressed="false">
            <span id="speaker-text" aria-hidden="true">&#128264;</span>
        </button>
    </div>
    <div id="dummyFocus" tabindex="-1" style="position:absolute; left:-9999px;"></div>
    <?php
    return ob_get_clean();
}
add_shortcode('speaker', 'speaker_button_shortcode');

/* -------------------------------------------------------------
 * Frontend Scripts & Styles
 * ------------------------------------------------------------- */
function speak_easy_scripts() {
    wp_enqueue_script(
        'speech-js',
        plugin_dir_url(__FILE__) . 'js/speaker.min.js',
        array(),
        '1.1',
        true
    );
    wp_enqueue_style(
        'speak-easy-default-css',
        plugin_dir_url(__FILE__) . 'css/speakeasy.css'
    );
    $custom_css = speak_easy_get_custom_css();
    if (!empty(trim($custom_css))) {
        wp_add_inline_style('speak-easy-default-css', $custom_css);
    }
    $full_lang = get_option('speak_easy_lang_source', 'de-DE');
    $lang_source = strtolower(substr(explode('-', $full_lang)[0], 0, 2));
    $bilder_erfassen = (bool) get_option('speak_easy_bilder_erfassen', false);
    $word_replacements = get_option('speak_easy_word_replacements', array());
    wp_localize_script('speech-js', 'SpeakEasySettings', array(
        'langSource'       => $lang_source,
        'bilderErfassen'   => $bilder_erfassen,
        'wordReplacements' => $word_replacements,
    ));
}
add_action('wp_enqueue_scripts', 'speak_easy_scripts');

/* -------------------------------------------------------------
 * Admin Menü & Settings
 * ------------------------------------------------------------- */
function vorlesen_add_admin_menu() {
    add_options_page(
        __('Speak Easy', 'speak-easy'),
        __('Speak Easy', 'speak-easy'),
        'manage_options',
        'speak-easy-css',
        'speak_easy_options_page'
    );
}
add_action('admin_menu', 'vorlesen_add_admin_menu');

function vorlesen_settings_init() {
    register_setting('speak_easy_settings_group', 'speak_easy_custom_css');
    register_setting('speak_easy_settings_group', 'speak_easy_button_title');
    register_setting('speak_easy_settings_group', 'speak_easy_lang_source', array(
        'sanitize_callback' => 'speak_easy_sanitize_lang',
    ));
    register_setting('speak_easy_settings_group', 'speak_easy_bilder_erfassen');
}
add_action('admin_init', 'vorlesen_settings_init');

function speak_easy_sanitize_lang($value) {
    $value = trim($value);
    if (empty($value)) return 'de-DE';
    $parts = explode('-', $value);
    $parts[0] = strtolower(substr($parts[0], 0, 2)) . substr($parts[0], 2);
    return implode('-', $parts);
}

/* -------------------------------------------------------------
 * Admin Handler (Save/Import/Export/Reset)
 * ------------------------------------------------------------- */
add_action('admin_init', 'speak_easy_admin_handlers');
function speak_easy_admin_handlers() {
    if (!current_user_can('manage_options')) return;

    // Speichern Wortersetzungen
    if (isset($_POST['save_word_replacements'])) {
        check_admin_referer('speak_easy_word_replacements', 'speak_easy_word_replacements_nonce');
        $keys = $_POST['speak_easy_word_replacements_keys'] ?? [];
        $values = $_POST['speak_easy_word_replacements_values'] ?? [];
        $clean = [];
        foreach ($keys as $i => $key) {
            $key = trim($key);
            $value = trim($values[$i] ?? '');
            if ($key !== '') $clean[$key] = $value;
        }
        update_option('speak_easy_word_replacements', $clean);
        add_action('admin_notices', function() {
            echo '<div class="updated"><p>' . esc_html(__('Word replacements saved.', 'speak-easy')) . '</p></div>';
        });
    }

    // Import
    if (isset($_POST['import_replacements'])) {
        check_admin_referer('speak_easy_import', 'speak_easy_import_nonce');
        if (!empty($_FILES['import_file']['tmp_name'])) {
            $json = file_get_contents($_FILES['import_file']['tmp_name']);
            $data = json_decode($json, true);
            $clean = [];
            if (is_array($data)) {
                foreach ($data as $k => $v) $clean[trim($k)] = trim($v);
            }
            update_option('speak_easy_word_replacements', $clean);
            add_action('admin_notices', function() {
                echo '<div class="updated"><p>' . esc_html(__('Word replacements imported successfully.', 'speak-easy')) . '</p></div>';
            });
        }
    }

    // Export
    if (isset($_POST['export_replacements'])) {
        check_admin_referer('speak_easy_export', 'speak_easy_export_nonce');
        $data = get_option('speak_easy_word_replacements', array());
        header('Content-Description: File Transfer');
        header('Content-Disposition: attachment; filename=speak-easy-replacements.json');
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
    }

    // Restore Defaults
    if (isset($_POST['restore_defaults'])) {
        check_admin_referer('speak_easy_restore', 'speak_easy_restore_nonce');
        $default_file = plugin_dir_path(__FILE__) . 'replacements-default.json';
        if (file_exists($default_file)) {
            $json = file_get_contents($default_file);
            $data = json_decode($json, true);
            $clean = [];
            if (is_array($data)) {
                foreach ($data as $k => $v) $clean[trim($k)] = trim($v);
            }
            update_option('speak_easy_word_replacements', $clean);
            add_action('admin_notices', function() {
                echo '<div class="updated"><p>' . esc_html(__('Word replacements reset to default.', 'speak-easy')) . '</p></div>';
            });
        }
    }

    // CSS Reset
    if (isset($_POST['reset_css'])) {
        check_admin_referer('speak_easy_reset_css', 'speak_easy_reset_nonce');
        $css_file = plugin_dir_path(__FILE__) . 'css/speakeasy.css';
        $default_css = file_exists($css_file) ? file_get_contents($css_file) : '';
        update_option('speak_easy_custom_css', $default_css);
        add_action('admin_notices', function() {
            echo '<div class="updated"><p>' . esc_html(__('Default CSS restored.', 'speak-easy')) . '</p></div>';
        });
    }
}

/* -------------------------------------------------------------
 * Admin Optionen-Seite
 * ------------------------------------------------------------- */
function speak_easy_options_page() {
    if (!current_user_can('manage_options')) return;

    $custom_css   = speak_easy_get_custom_css();
    $button_title = get_option('speak_easy_button_title') ?: __('Read text / Stop reading', 'speak-easy');
    $lang_source  = get_option('speak_easy_lang_source', 'de-DE');
    $bilder_erfassen = get_option('speak_easy_bilder_erfassen', 0);
    $replacements = get_option('speak_easy_word_replacements', array());
    ?>
    <div class="wrap">
        <h1><?php echo esc_html(__('Speak Easy Plugin – Settings', 'speak-easy')); ?></h1>

        <!-- Hauptoptionen -->
        <form method="post" action="options.php">
            <?php settings_fields('speak_easy_settings_group'); do_settings_sections('speak_easy_settings_group'); ?>

            <h2>🌐 <?php _e('Language of HTML texts', 'speak-easy'); ?></h2>
            <p><?php _e('Enter the language of your website in ISO 639 format, e.g., de-DE, en-US, en-GB, fr-FR, fr-BE, zh-CN. Short forms like de, en, fr, zh also work.', 'speak-easy'); ?></p>
            <input type="text" name="speak_easy_lang_source" value="<?php echo esc_attr($lang_source); ?>" class="regular-text" />

            <h2>🔊 <?php _e('Title of the speaker button', 'speak-easy'); ?></h2>
            <input type="text" name="speak_easy_button_title" value="<?php echo esc_attr($button_title); ?>" class="regular-text" />

            <h2>🖼️ <?php _e('Capturing images linguistically', 'speak-easy'); ?></h2>
            <label>
                <input type="checkbox" name="speak_easy_bilder_erfassen" value="1" <?php checked(1, $bilder_erfassen); ?>>
                <?php _e('Read image content (alt and title, if present)', 'speak-easy'); ?>
            </label>

            <h2>🎨 <?php _e('Custom CSS', 'speak-easy'); ?></h2>
            <textarea name="speak_easy_custom_css" rows="10" cols="80" class="large-text code"><?php echo esc_textarea($custom_css); ?></textarea>

            <?php submit_button(__('Save settings', 'speak-easy')); ?>
        </form>

        <!-- CSS Reset -->
        <form method="post" action="" style="margin-top:10px;">
            <?php wp_nonce_field('speak_easy_reset_css', 'speak_easy_reset_nonce'); ?>
            <input type="hidden" name="reset_css" value="1">
            <?php submit_button(__('Restore default CSS', 'speak-easy'), 'secondary'); ?>
        </form>

        <!-- Wortersetzung -->
        <h2>🔤 <?php _e('Word replacements for speech output', 'speak-easy'); ?></h2>
		<p><?php echo esc_html(__('In the left column, you can enter terms or words that will be replaced by the expressions in the right column when the text is read aloud. This way, for example, abbreviations can be automatically replaced with their full text.', 'speak-easy')); ?></p>
		<form method="post" action="">
            <?php wp_nonce_field('speak_easy_word_replacements', 'speak_easy_word_replacements_nonce'); ?>
            <table id="word-replacement-table" class="widefat">
                <thead>
                    <tr>
                        <th><?php _e('Original word', 'speak-easy'); ?></th>
                        <th><?php _e('Replace by', 'speak-easy'); ?></th>
                        <th><?php _e('Action', 'speak-easy'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($replacements as $original => $replacement): ?>
                        <tr>
                            <td><input type="text" name="speak_easy_word_replacements_keys[]" value="<?php echo esc_attr($original); ?>"></td>
                            <td><input type="text" name="speak_easy_word_replacements_values[]" value="<?php echo esc_attr($replacement); ?>"></td>
                            <td><button type="button" class="remove-row button">−</button></td>
                        </tr>
                    <?php endforeach; ?>
                    <tr>
                        <td><input type="text" name="speak_easy_word_replacements_keys[]"></td>
                        <td><input type="text" name="speak_easy_word_replacements_values[]"></td>
                        <td><button type="button" class="remove-row button">−</button></td>
                    </tr>
                </tbody>
            </table>
            <p>
                <button type="button" id="add-word-row" class="button">+ <?php _e('New line', 'speak-easy'); ?></button>
            </p>
            <button type="submit" name="save_word_replacements" class="button button-primary"><?php _e('Save word replacement table', 'speak-easy'); ?></button>
        </form>

        <!-- Import / Export / Restore -->
        <h2>⬇️⬆️ <?php _e('Import / Export word replacement table', 'speak-easy'); ?></h2>

        <form method="post" enctype="multipart/form-data">
            <?php wp_nonce_field('speak_easy_import', 'speak_easy_import_nonce'); ?>
            <input type="file" name="import_file" accept="application/json" required>
            <button type="submit" name="import_replacements" class="button"><?php _e('Import (JSON)', 'speak-easy'); ?></button>
        </form>

        <form method="post" style="margin-top:10px;">
            <?php wp_nonce_field('speak_easy_export', 'speak_easy_export_nonce'); ?>
            <button type="submit" name="export_replacements" class="button"><?php _e('Export (JSON)', 'speak-easy'); ?></button>
        </form>

        <form method="post" style="margin-top:10px;">
            <?php wp_nonce_field('speak_easy_restore', 'speak_easy_restore_nonce'); ?>
            <button type="submit" name="restore_defaults" class="button"><?php _e('Restore default word table', 'speak-easy'); ?></button>
        </form>

        <!-- Preview -->
        <h2>🧪 <?php _e('Preview of word replacements', 'speak-easy'); ?></h2>
        <p class="description" style="font-style: italic; color: #555;">
        <?php _e('The «Convert Text» button prepares any entered text – taking the word replacement table into account – for speech output. «Read aloud» will speak the text if the language is supported by the browser.', 'speak-easy'); ?>
        </p>
        <textarea id="preview-input" rows="4" cols="80" class="large-text code" placeholder="<?php _e('Enter sample text here...', 'speak-easy'); ?>"></textarea>
        <p>
            <button type="button" id="preview-button" class="button"><?php _e('Convert text', 'speak-easy'); ?></button>
            <button type="button" id="preview-speak" class="button">🔊 <?php _e('Read aloud', 'speak-easy'); ?></button>
        </p>
        <h3>🔁 <?php _e('Result after word replacements:', 'speak-easy'); ?></h3>
        <div id="preview-output" style="border:1px solid #ccc; padding:10px; background:#f9f9f9;"></div>
    </div>

    <script>
    document.getElementById('add-word-row').addEventListener('click', function () {
        const table = document.getElementById('word-replacement-table').getElementsByTagName('tbody')[0];
        const newRow = document.createElement('tr');
        newRow.innerHTML = `
            <td><input type="text" name="speak_easy_word_replacements_keys[]" /></td>
            <td><input type="text" name="speak_easy_word_replacements_values[]" /></td>
            <td><button type="button" class="remove-row button">−</button></td>
        `;
        table.appendChild(newRow);
    });

    document.addEventListener('click', function (e) {
        if (e.target && e.target.classList.contains('remove-row')) {
            const row = e.target.closest('tr');
            row.parentNode.removeChild(row);
        }
    });

    document.getElementById('preview-button').addEventListener('click', function () {
        const input = document.getElementById('preview-input').value;
        const replacements = <?php echo json_encode(get_option('speak_easy_word_replacements', [])); ?>;
        let output = input;
        for (const [original, replacement] of Object.entries(replacements)) {
            const regex = new RegExp('\\b' + original + '\\b', 'gi');
            output = output.replace(regex, replacement);
        }
        document.getElementById('preview-output').textContent = output;
    });

    document.getElementById('preview-speak').addEventListener('click', function () {
        const text = document.getElementById('preview-output').textContent;
        if ('speechSynthesis' in window) {
            const utterance = new SpeechSynthesisUtterance(text);
            utterance.lang = '<?php echo esc_js(get_option("speak_easy_lang_source", "de-DE")); ?>';
            speechSynthesis.speak(utterance);
        } else {
            alert('<?php echo esc_js(__('Speech output is not supported by this browser.', 'speak-easy')); ?>');
        }
    });
    </script>
<?php
}
?>